<?php

namespace App;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Carbon\Carbon;

class Carta extends Model
{
    use SoftDeletes;

    protected $table = 'carta';
    protected $fillable = [
    	//1. Número
		'numero',
		//2. Nombre y dirección del remitente
		'remitente_nombre',
        'remitente_direccion',
		//3. Nombre y dirección del destinatario
		'destinatario_nombre',
        'destinatario_direccion',
		//4. Nombre y dirección del consignatario
		'consignatario_nombre',
        'consignatario_direccion',
		//5. Notificar a
		'notificar_a',
		//6. Lugar, país y fecha en que el transportista recibe las mercancías
		'recepcion_lugar',
       	'recepcion_pais_id',
       	'recepcion_fecha',
		//7. Lugar, país y fecha de embarque de las mercancías
		'embarque_lugar',
        'embarque_pais_id',
        'embarque_fecha',
		//8. Lugar, país y fecha convenida para la entrega de las mercancías
		'entrega_lugar',
        'entrega_pais_id',
        'entrega_fecha',
		//9. Condiciones del transporte y condiciones de pago
		'condicion_pago',
		//10. Cantidad y clase de los bultos
		//11. Marcas y números de los bultos
		//12. Descripción corriente de la naturaleza de las mercancías (indicar si son peligrosas)
		//ESTO VA EN DETALLES
	
		//13. Peso en Kilogramos
		'peso_neto',
        'peso_bruto',
		//14. Volumen en metros cúbicos
		'peso_volumen',
		//15. Otras unidades de medida
		'peso_otras_unidades',
		//16. Precio de las mercancías (INCOTERMS 1990) y tipo de moneda
		'peso_tipo_moneda',
        'peso_precio_mercancia',
		//17. Gastos a Pagar
        'gasto_moneda',
		'gasto_flete_remitente_cargo',
        'gasto_flete_destinatario_cargo',
        'gasto_otro_remitente_cargo',
        'gasto_otro_destinatario_cargo',
		//18. Documentos recibidos del remitente
		'remitente_documentos_recibidos',
		//19. Lugar, país y fecha de emisión
		'emision_lugar',
        'emision_pais_id',
        'emision_fecha',
		//20. Nombre y firma del remitente o su representante o agente
		//PDF
		//21. Instrucciones al transportista
		'transportista_instrucciones',
		//22. Observaciones del transportista
		'transportista_observaciones',	
		//23. Nombre, firma y sello del transportista autorizado o su representante o su agente
		//PDF

		'bodega_imprime_manifesto',
        'public'
    ];

    protected $appends = [
        'numero_desc',

        'gasto_total_remitente',
        'gasto_total_destinatario',

        'iso_recepcion_fecha',
        'iso_embarque_fecha',
        'iso_entrega_fecha',
        'iso_emision_fecha',
    ];

	protected $casts = [
		'peso_neto'                      => 'float',
		'peso_bruto'                     => 'float',
		'peso_volumen'                   => 'float',
		'peso_otras_unidades'            => 'float',
		'peso_precio_mercancia'          => 'float',
		'gasto_flete_destinatario_cargo' => 'float',
		'gasto_flete_remitente_cargo'    => 'float',
		'gasto_otro_remitente_cargo'     => 'float',
		'gasto_otro_destinatario_cargo'  => 'float',

        'public' => 'boolean',
    ];

    public function detalles()
    {
    	return $this->hasMany(CartaDetalle::class);
    }

    public function recepcionPais()
    {
    	return $this->belongsTo(Pais::class,'recepcion_pais_id');
    }

    public function embarquePais()
    {
    	return $this->belongsTo(Pais::class,'embarque_pais_id');
    }

    public function entregaPais()
    {
    	return $this->belongsTo(Pais::class,'entrega_pais_id');
    }

    public function emisionPais()
    {
    	return $this->belongsTo(Pais::class,'emision_pais_id');
    }

    /**
     * Appends
     */
    
    public function getNumeroDescAttribute()
    {
        return str_pad($this->numero, 7, '0', STR_PAD_LEFT);
    }

    public function getGastoTotalRemitenteAttribute()
    {
    	return $this->gasto_flete_remitente_cargo + $this->gasto_otro_remitente_cargo;
    }

    public function getGastoTotalDestinatarioAttribute()
    {
    	return $this->gasto_flete_destinatario_cargo + $this->gasto_otro_destinatario_cargo;
    }

    public function getIsoRecepcionFechaAttribute()
    {
        return static::DateFormat($this->recepcion_fecha);
    }

    public function getIsoEmbarqueFechaAttribute()
    {
        return static::DateFormat($this->embarque_fecha);
    }

    public function getIsoEntregaFechaAttribute()
    {
        return static::DateFormat($this->entrega_fecha);
    }

    public function getIsoEmisionFechaAttribute()
    {
        return static::DateFormat($this->emision_fecha);
    }

    /**
     * helpers
     */
    
    public static function DateFormat($date)
    {
        if(is_null($date)) return '';
        
        $date_carbon = new Carbon($date);
        $iso_month = $date_carbon->isoFormat('MMMM');
        return "{$date_carbon->day} de {$iso_month} del {$date_carbon->year}";
    }

}
